﻿#if UNITY_IOS
using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using Unity.Notifications.iOS;
using System;
using System.Runtime.InteropServices;
using AOT;
using System.Threading;

namespace Aliyun.Push.IOS
{
    public class iOSPushHelper
    {
        private static Dictionary<int, Action<bool, string>> CallbackPairs = new Dictionary<int, Action<bool, string>>();
        private static int ActionIndex = 0;
        private static Action<iOSNotification> customNotificationReceivedCallback;
        private static Action<string, string> customMessageReceivedCallback;

        private static IEnumerator RequestAuthorization(Action<bool, string> callback)
        {
            var authorizationOption = AuthorizationOption.Alert | AuthorizationOption.Badge;
            using (var req = new AuthorizationRequest(authorizationOption, true))
            {
                while (!req.IsFinished)
                {
                    yield return null;
                };

                string res = "\n RequestAuthorization:";
                res += "\n finished: " + req.IsFinished;
                res += "\n granted :  " + req.Granted;
                res += "\n error:  " + req.Error;
                res += "\n deviceToken:  " + req.DeviceToken;
                Debug.Log(res);
                if (req.DeviceToken == null)
                {
                    callback(false, "deviceToken is null");
                }
                else
                {
                    int actionId = Interlocked.Increment(ref ActionIndex);
                    CallbackPairs.Add(actionId, callback);
                    _reportToken(req.DeviceToken, actionId);
                    iOSNotificationCenter.OnRemoteNotificationReceived += NotificationReceivedCallback;
                }
            }
        }

        public static void Init(string appKey, string appSecret, Action<bool, string> callback)
        {
            _setCloudPushCallback(CloudPushCallbackReceived);
            _setMessageCallback(MessageReceived);
            int actionId = Interlocked.Increment(ref ActionIndex);
            CallbackPairs.Add(actionId, callback);
            _initCloudPush(appKey, appSecret, actionId);
        }

        public static void RegisterDeviceToken(MonoBehaviour mono, Action<bool, string> callback)
        {
            mono.StartCoroutine(RequestAuthorization(callback));
        }



        public static void BindAccount(string account, Action<bool, string> callback)
        {
            int actionId = Interlocked.Increment(ref ActionIndex);
            CallbackPairs.Add(actionId, callback);
            _bindAccount(account, actionId);
        }

        public static void UnBindAccount(Action<bool, string> callback)
        {
            int actionId = Interlocked.Increment(ref ActionIndex);
            CallbackPairs.Add(actionId, callback);
            _unBindAccount(actionId);
        }

        public static void BindTag(int target, string tag, string alias, Action<bool, string> callback)
        {
            int actionId = Interlocked.Increment(ref ActionIndex);
            CallbackPairs.Add(actionId, callback);
            _bindTag(target, tag, alias, actionId);
        }

        public static void UnBindTag(int target, string tag, string alias, Action<bool, string> callback)
        {
            int actionId = Interlocked.Increment(ref ActionIndex);
            CallbackPairs.Add(actionId, callback);
            _unBindTag(target, tag, alias, actionId);
        }

        public static void ListTag(int target, Action<bool, string> callback)
        {
            int actionId = Interlocked.Increment(ref ActionIndex);
            CallbackPairs.Add(actionId, callback);
            _listTag(target, actionId);
        }


        public static void AddAlias(string alias, Action<bool, string> callback)
        {
            int actionId = Interlocked.Increment(ref ActionIndex);
            CallbackPairs.Add(actionId, callback);
            _addAlias(alias, actionId);
        }

        public static void RemoveAlias(string alias, Action<bool, string> callback)
        {
            int actionId = Interlocked.Increment(ref ActionIndex);
            CallbackPairs.Add(actionId, callback);
            _removeAlias(alias, actionId);
        }

        public static void ListAlias(Action<bool, string> callback)
        {
            int actionId = Interlocked.Increment(ref ActionIndex);
            CallbackPairs.Add(actionId, callback);
            _listAlias(actionId);
        }

        public static string DeviceId()
        {
            return _deviceId();
        }

        public static void SetNotificationReceivedCallback(Action<iOSNotification> notificationReceivedCallback)
        {
            customNotificationReceivedCallback = notificationReceivedCallback;
        }

        public static void SetMessageReceivedCallback(Action<string, string> messageReceivedCallback)
        {
            customMessageReceivedCallback = messageReceivedCallback;
        }



        private static void NotificationReceivedCallback(iOSNotification notification)
        {
            if (notification != null)
            {
                var msg = "Received Notification: " + notification.Identifier;
                msg += "\n - Notification received: ";
                msg += "\n - .Title: " + notification.Title;
                msg += "\n - .Badge: " + notification.Badge;
                msg += "\n - .Body: " + notification.Body;
                msg += "\n - .CategoryIdentifier: " + notification.CategoryIdentifier;
                msg += "\n - .Subtitle: " + notification.Subtitle;
                msg += "\n - .Data: " + notification.Data;
                Debug.Log(msg);
            }

            if (customNotificationReceivedCallback != null)
            {
                customNotificationReceivedCallback(notification);
            }
        }

        [DllImport("__Internal")]
        private static extern void _setCloudPushCallback(CloudPushCallback callback);

        [DllImport("__Internal")]
        private static extern void _setMessageCallback(MessageCallback callback);

        [DllImport("__Internal")]
        private static extern void _initCloudPush(string appKey, string appSecret, int actionId);

        [DllImport("__Internal")]
        private static extern void _reportToken(string deviceToken, int actionId);

        [DllImport("__Internal")]
        private static extern void _bindAccount(string account, int actionId);

        [DllImport("__Internal")]
        private static extern void _unBindAccount(int actionId);

        [DllImport("__Internal")]
        private static extern void _bindTag(int target, string tag, string alias, int actionId);

        [DllImport("__Internal")]
        private static extern void _unBindTag(int target, string tag, string alias, int actionId);

        [DllImport("__Internal")]
        private static extern void _listTag(int target, int actionId);

        [DllImport("__Internal")]
        private static extern void _addAlias(string alias, int actionId);

        [DllImport("__Internal")]
        private static extern void _removeAlias(string alias, int actionId);

        [DllImport("__Internal")]
        private static extern void _listAlias(int actionId);

        [DllImport("__Internal")]
        private static extern string _deviceId();

        private delegate void CloudPushCallback(int actionId, bool success, string msg);

        private delegate void MessageCallback(string title, string content);

        [MonoPInvokeCallback(typeof(CloudPushCallback))]
        public static void CloudPushCallbackReceived(int actionId, bool success, string msg)
        {
            Debug.Log(success + " " + msg);
            // call cs code
            Action<bool, string> callback = CallbackPairs[actionId];
            if (callback != null)
            {
                callback(success, msg);
                CallbackPairs.Remove(actionId);
            }
        }

        [MonoPInvokeCallback(typeof(MessageCallback))]
        public static void MessageReceived(string title, string content)
        {
            Debug.Log(title + " " + content);
            // call cs code
            if(customMessageReceivedCallback != null)
            {
                customMessageReceivedCallback(title, content);
            }
        }


    }
}
#endif