//
//  PushWrapper.m
//  Unity-iPhone
//
//  Created by 心定 on 2021/3/25.
//

#import <Foundation/Foundation.h>
#import <CloudPushSDK/CloudPushSDK.h>
#import "PushWrapper.h"

@implementation PushWrapper

const int REGISTER = 1;
const int BIND_ACCOUNT = 2;
const int UNBIND_ACCOUNT = 3;
const int BIND_TAG = 4;
const int UNBIND_TAG = 5;
const int LIST_TAG = 6;
const int ADD_ALIAS = 7;
const int REMOVE_ALIAS = 8;
const int LIST_ALIAS = 9;

const int BIND_PHONE = 10;
const int UNBIND_PHONE = 11;

const int REPORT_TOKEN = 12;

+ (instancetype)sharedInstance
{
    static PushWrapper *sharedInstance = nil;
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^{
        sharedInstance = [[PushWrapper alloc] init];
    });

    return sharedInstance;
}

- (void)invokeCallback:(CloudPushCallbackResult *)res actionId:(int)actionId {
    if(self.onPushActionCallback != NULL) {
        if(res.success) {
            if(res.data != NULL){
                self.onPushActionCallback(actionId, res.success, [[NSString stringWithFormat:@"%@", res.data] UTF8String]);
            } else {
                self.onPushActionCallback(actionId, res.success, nil);
            }
        } else {
            self.onPushActionCallback(actionId, res.success, [[NSString stringWithFormat:@"%@", res.error] UTF8String]);
        }
    }
}

- (NSString *)deviceId {
    return [CloudPushSDK getDeviceId];
}

- (void)initCloudPush:(NSString *)appKey appSecret:(NSString *)appSecret actionId:(int)actionId {
    // 正式上线建议关闭
    [CloudPushSDK turnOnDebug];
    // SDK初始化，手动输出appKey和appSecret
    [CloudPushSDK asyncInit:appKey appSecret:appSecret callback:^(CloudPushCallbackResult *res) {
        if (res.success) {
            NSLog(@"Push SDK init success, deviceId: %@.", [CloudPushSDK getDeviceId]);
            [self registerMessageReceive];
        } else {
            NSLog(@"Push SDK init failed, error: %@", res.error);
        }
        [self invokeCallback:res actionId:actionId];
    }];
}

- (void)reportToken:(NSString *)deviceToken actionId:(int)actionId {
    NSLog(@"Upload deviceToken to CloudPush server.");
    NSData * data = [self dataFromHexString:deviceToken];
    [CloudPushSDK registerDevice:data withCallback:^(CloudPushCallbackResult *res) {
        if (res.success) {
            NSLog(@"Register deviceToken success, deviceToken: %@", [CloudPushSDK getApnsDeviceToken]);
        } else {
            NSLog(@"Register deviceToken failed, error: %@", res.error);
        }
        [self invokeCallback:res actionId:actionId];
    }];
}

- (NSData *)dataFromHexString:(NSString *)hexstr {
    NSMutableData *data = [[NSMutableData alloc] init];
    NSString *inputStr = [hexstr uppercaseString];

    NSString *hexChars = @"0123456789ABCDEF";

    Byte b1, b2;
    b1 = 255;
    for (int i = 0; i < hexstr.length; i ++) {
        NSString *subStr = [inputStr substringWithRange:NSMakeRange(i, 1)];
        NSRange loc = [hexChars rangeOfString:subStr];

        if (loc.location == NSNotFound) continue;

        if (255 == b1) {
            b1 = (Byte)loc.location;
        }else {
            b2 = (Byte)loc.location;

            // Appending the Byte to NSData
            Byte *bytes = malloc(sizeof(Byte) * 1);
            bytes[0] = ((b1 << 4) & 0xf0) | (b2 & 0x0f);
            [data appendBytes:bytes length:1];

            b1 = b2 = 255;
        }
    }

    return data;
}

- (void)bindAccount:(NSString *)account actionId:(int)actionId {
    [CloudPushSDK bindAccount:account withCallback:^(CloudPushCallbackResult *res) {
        [self invokeCallback:res actionId:actionId];
    }];
}

- (void)unBindAccount:(int)actionId {
    [CloudPushSDK unbindAccount:^(CloudPushCallbackResult *res) {
        [self invokeCallback:res actionId:actionId];
    }];
}

- (void)bindTag:(int)target
       withTag:(NSString *)tag
      withAlias:(NSString *)alias
       actionId:(int)actionId{
    NSArray *tagArray = [[NSArray alloc] initWithObjects:tag, nil];
    [CloudPushSDK bindTag:target withTags:tagArray withAlias:alias withCallback:^(CloudPushCallbackResult *res) {
        [self invokeCallback:res actionId:actionId];
    }];
}

- (void)unBindTag:(int)target
         withTag:(NSString *)tag
        withAlias:(NSString *)alias
         actionId:(int)actionId {
    NSArray *tagArray = [[NSArray alloc] initWithObjects:tag, nil];
    [CloudPushSDK unbindTag:target withTags:tagArray withAlias:alias withCallback:^(CloudPushCallbackResult *res) {
        [self invokeCallback:res actionId:actionId];
    }];
}

- (void)listTags:(int)target
        actionId:(int)actionId {
    [CloudPushSDK listTags:target withCallback:^(CloudPushCallbackResult *res) {
        [self invokeCallback:res actionId:actionId];
    }];
}

- (void)addAlias:(NSString *)alias
        actionId:(int)actionId {
    [CloudPushSDK addAlias:alias withCallback:^(CloudPushCallbackResult *res) {
        [self invokeCallback:res actionId:actionId];
    }];
}

- (void)removeAlias:(NSString *)alias
           actionId:(int)actionId{
    [CloudPushSDK removeAlias:alias withCallback:^(CloudPushCallbackResult *res) {
        [self invokeCallback:res actionId:actionId];
    }];
}

- (void)listAliases:(int)actionId {
    [CloudPushSDK listAliases:^(CloudPushCallbackResult *res) {
        [self invokeCallback:res actionId:actionId];
    }];
}


/**
 *    @brief    注册推送消息到来监听
 */
- (void)registerMessageReceive {
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(onMessageReceived:)
                                                 name:@"CCPDidReceiveMessageNotification"
                                               object:nil];
}

/**
 *    处理到来推送消息
 *
 *    @param     notification
 */
- (void)onMessageReceived:(NSNotification *)notification {
    NSLog(@"Receive one message!");
   
    CCPSysMessage *message = [notification object];
    NSString *title = [[NSString alloc] initWithData:message.title encoding:NSUTF8StringEncoding];
    NSString *body = [[NSString alloc] initWithData:message.body encoding:NSUTF8StringEncoding];
    NSLog(@"Receive message title: %@, content: %@.", title, body);
    
    if(self.onMessageReceivedCallback != NULL){
        self.onMessageReceivedCallback([title UTF8String], [body UTF8String]);
    }
}





@end
